//
//  PerformanceTests.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import XCTest
@testable import death_app_Watch_App

final class PerformanceTests: XCTestCase {
    
    var actuarialModel: ActuarialModel!
    var predictionEngine: PredictionEngine!
    var coreDataManager: CoreDataManager!
    
    override func setUpWithError() throws {
        super.setUp()
        actuarialModel = ActuarialModel()
        predictionEngine = PredictionEngine()
        coreDataManager = CoreDataManager.shared
    }
    
    override func tearDownWithError() throws {
        actuarialModel = nil
        predictionEngine = nil
        coreDataManager = nil
        super.tearDown()
    }
    
    // MARK: - Calculation Speed Tests
    
    func testBasicLifeExpectancyCalculationSpeed() throws {
        let riskFactors: [String: Double] = [
            "bmi": 25.0,
            "smoking": 0.0,
            "exercise": 180.0,
            "alcohol": 0.5
        ]
        
        measure {
            for age in stride(from: 20.0, through: 80.0, by: 1.0) {
                _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors)
            }
        }
    }
    
    func testComplexRiskFactorCalculationSpeed() throws {
        let complexRiskFactors: [String: Double] = [
            "bmi": 28.5,
            "smoking": 1.0,
            "exercise": 45.0,
            "alcohol": 2.0,
            "stress": 0.7,
            "sleep": 6.5,
            "diet": 0.6,
            "familyHistory": 0.8,
            "bloodPressure": 140.0,
            "cholesterol": 220.0
        ]
        
        measure {
            for age in stride(from: 25.0, through: 75.0, by: 5.0) {
                _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: complexRiskFactors)
            }
        }
    }
    
    func testGompertzMakehamCalculationSpeed() throws {
        measure {
            for age in stride(from: 0.0, through: 120.0, by: 0.1) {
                _ = actuarialModel.calculateGompertzMakeham(age: age)
            }
        }
    }
    
    func testCoxProportionalHazardsSpeed() throws {
        let riskFactors: [String: Double] = [
            "smoking": 1.0,
            "exercise": 0.3,
            "bmi": 32.0,
            "alcohol": 1.5,
            "stress": 0.8
        ]
        
        measure {
            for _ in 0..<10000 {
                _ = actuarialModel.calculateCoxHazardRatio(riskFactors: riskFactors)
            }
        }
    }
    
    // MARK: - Database Query Performance Tests
    
    func testPredictionSavePerformance() throws {
        let predictions = generateTestPredictions(count: 1000)
        
        measure {
            for prediction in predictions {
                coreDataManager.savePrediction(prediction)
            }
        }
    }
    
    func testPredictionFetchPerformance() throws {
        // First, save some predictions
        let predictions = generateTestPredictions(count: 1000)
        for prediction in predictions {
            coreDataManager.savePrediction(prediction)
        }
        
        measure {
            _ = coreDataManager.fetchAllPredictions()
        }
    }
    
    func testPredictionQueryPerformance() throws {
        // Save predictions with different dates
        let predictions = generateTestPredictions(count: 5000)
        for prediction in predictions {
            coreDataManager.savePrediction(prediction)
        }
        
        let startDate = Calendar.current.date(byAdding: .month, value: -1, to: Date())!
        let endDate = Date()
        
        measure {
            _ = coreDataManager.fetchPredictions(from: startDate, to: endDate)
        }
    }
    
    func testBatchPredictionOperations() throws {
        let largeBatchPredictions = generateTestPredictions(count: 10000)
        
        measure {
            coreDataManager.batchSavePredictions(largeBatchPredictions)
        }
    }
    
    func testPredictionDeletionPerformance() throws {
        // First save many predictions
        let predictions = generateTestPredictions(count: 2000)
        for prediction in predictions {
            coreDataManager.savePrediction(prediction)
        }
        
        measure {
            coreDataManager.deleteAllPredictions()
        }
    }
    
    // MARK: - Memory Usage Tests
    
    func testMemoryUsageUnderLoad() throws {
        let initialMemory = getCurrentMemoryUsage()
        
        // Generate many predictions
        var predictions: [LifePrediction] = []
        for _ in 0..<1000 {
            let prediction = generateRandomPrediction()
            predictions.append(prediction)
        }
        
        let peakMemory = getCurrentMemoryUsage()
        
        // Clear predictions
        predictions.removeAll()
        
        let finalMemory = getCurrentMemoryUsage()
        
        XCTAssertLessThan(finalMemory - initialMemory, peakMemory - initialMemory, "Memory should be released after clearing")
        XCTAssertLessThan(peakMemory - initialMemory, 100 * 1024 * 1024, "Peak memory increase should be reasonable") // 100MB limit
    }
    
    func testMemoryLeaksInCalculations() throws {
        let initialMemory = getCurrentMemoryUsage()
        
        // Perform many calculations
        for _ in 0..<10000 {
            let age = Double.random(in: 20...80)
            let riskFactors = generateRandomRiskFactors()
            _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors)
        }
        
        let finalMemory = getCurrentMemoryUsage()
        
        // Memory increase should be minimal for calculations
        XCTAssertLessThan(finalMemory - initialMemory, 10 * 1024 * 1024, "Calculations should not leak significant memory") // 10MB limit
    }
    
    // MARK: - Battery Impact Tests
    
    func testCPUUsageUnderLoad() throws {
        let startTime = CFAbsoluteTimeGetCurrent()
        var calculationCount = 0
        
        // Run calculations for 1 second
        let timeout = startTime + 1.0
        while CFAbsoluteTimeGetCurrent() < timeout {
            let age = Double.random(in: 25...75)
            let riskFactors = generateRandomRiskFactors()
            _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors)
            calculationCount += 1
        }
        
        // Should complete reasonable number of calculations per second
        XCTAssertGreaterThan(calculationCount, 100, "Should complete at least 100 calculations per second")
        XCTAssertLessThan(calculationCount, 100000, "Should not be unreasonably fast (indicates test issue)")
    }
    
    func testBackgroundProcessingEfficiency() throws {
        let expectation = XCTestExpectation(description: "Background processing")
        let startTime = Date()
        
        DispatchQueue.global(qos: .background).async {
            // Simulate background prediction generation
            let predictions = self.generateTestPredictions(count: 100)
            for prediction in predictions {
                self.coreDataManager.savePrediction(prediction)
            }
            
            let endTime = Date()
            let processingTime = endTime.timeIntervalSince(startTime)
            
            // Background processing should complete in reasonable time
            XCTAssertLessThan(processingTime, 5.0, "Background processing should complete within 5 seconds")
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 10.0)
    }
    
    // MARK: - Concurrent Operation Performance
    
    func testConcurrentCalculationPerformance() throws {
        let expectation = XCTestExpectation(description: "Concurrent calculations")
        expectation.expectedFulfillmentCount = 10
        
        let startTime = Date()
        
        for i in 0..<10 {
            DispatchQueue.global().async {
                let age = 30.0 + Double(i * 5)
                let riskFactors = self.generateRandomRiskFactors()
                
                for _ in 0..<100 {
                    _ = self.actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors)
                }
                
                expectation.fulfill()
            }
        }
        
        wait(for: [expectation], timeout: 10.0)
        
        let endTime = Date()
        let totalTime = endTime.timeIntervalSince(startTime)
        
        // Concurrent operations should complete faster than sequential
        XCTAssertLessThan(totalTime, 8.0, "Concurrent operations should complete within 8 seconds")
    }
    
    func testConcurrentDatabaseOperations() throws {
        let expectation = XCTestExpectation(description: "Concurrent database operations")
        expectation.expectedFulfillmentCount = 5
        
        for i in 0..<5 {
            DispatchQueue.global().async {
                let predictions = self.generateTestPredictions(count: 200)
                for prediction in predictions {
                    self.coreDataManager.savePrediction(prediction)
                }
                expectation.fulfill()
            }
        }
        
        wait(for: [expectation], timeout: 15.0)
        
        // Verify data integrity after concurrent operations
        let allPredictions = coreDataManager.fetchAllPredictions()
        XCTAssertEqual(allPredictions.count, 1000, "All predictions should be saved correctly")
    }
    
    // MARK: - Edge Case Performance
    
    func testPerformanceWithExtremeValues() throws {
        let extremeRiskFactors: [String: Double] = [
            "bmi": 60.0,
            "smoking": 10.0,
            "exercise": -100.0,
            "alcohol": 20.0,
            "unknownFactor": 999999.0
        ]
        
        measure {
            for age in [0.0, 150.0, -10.0, 1000.0] {
                _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: extremeRiskFactors)
            }
        }
    }
    
    func testPerformanceWithEmptyData() throws {
        measure {
            for age in stride(from: 20.0, through: 80.0, by: 1.0) {
                _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: [:])
            }
        }
    }
    
    // MARK: - Real-World Scenario Performance
    
    func testTypicalUserWorkflowPerformance() throws {
        let expectation = XCTestExpectation(description: "Typical user workflow")
        
        let startTime = Date()
        
        // Simulate typical user workflow
        let userProfile = UserProfile(
            birthDate: Calendar.current.date(byAdding: .year, value: -35, to: Date())!,
            gender: .male,
            height: 175.0,
            weight: 75.0
        )
        
        predictionEngine.generatePrediction(for: userProfile) { result in
            let endTime = Date()
            let processingTime = endTime.timeIntervalSince(startTime)
            
            switch result {
            case .success:
                // Typical workflow should complete quickly
                XCTAssertLessThan(processingTime, 2.0, "Typical prediction should complete within 2 seconds")
                
            case .failure:
                XCTFail("Typical workflow should succeed")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
    }
    
    func testAppLaunchPerformance() throws {
        measure {
            // Simulate app launch initialization
            let actuarialModel = ActuarialModel()
            let predictionEngine = PredictionEngine()
            let coreDataManager = CoreDataManager.shared
            
            // Load recent predictions
            _ = coreDataManager.fetchRecentPredictions(limit: 10)
            
            // Verify models are ready
            _ = actuarialModel.isModelReady()
            _ = predictionEngine.isReady()
        }
    }
    
    // MARK: - Helper Methods
    
    private func generateTestPredictions(count: Int) -> [LifePrediction] {
        var predictions: [LifePrediction] = []
        
        for i in 0..<count {
            let prediction = LifePrediction(
                lifeExpectancy: 70.0 + Double(i % 30),
                generatedAt: Date().addingTimeInterval(-Double(i * 3600)),
                riskFactors: generateRandomRiskFactors(),
                healthMetrics: generateRandomHealthMetrics(),
                confidenceScore: Double.random(in: 0.5...1.0)
            )
            predictions.append(prediction)
        }
        
        return predictions
    }
    
    private func generateRandomPrediction() -> LifePrediction {
        return LifePrediction(
            lifeExpectancy: Double.random(in: 60...95),
            generatedAt: Date().addingTimeInterval(-Double.random(in: 0...31536000)),
            riskFactors: generateRandomRiskFactors(),
            healthMetrics: generateRandomHealthMetrics(),
            confidenceScore: Double.random(in: 0.3...1.0)
        )
    }
    
    private func generateRandomRiskFactors() -> [String: Double] {
        return [
            "bmi": Double.random(in: 18...35),
            "smoking": Double.random(in: 0...1),
            "exercise": Double.random(in: 0...300),
            "alcohol": Double.random(in: 0...3),
            "stress": Double.random(in: 0...1)
        ]
    }
    
    private func generateRandomHealthMetrics() -> [String: Double] {
        return [
            "heartRate": Double.random(in: 60...100),
            "steps": Double.random(in: 3000...15000),
            "activeEnergy": Double.random(in: 200...800),
            "sleepHours": Double.random(in: 4...10)
        ]
    }
    
    private func getCurrentMemoryUsage() -> Int64 {
        var info = mach_task_basic_info()
        var count = mach_msg_type_number_t(MemoryLayout<mach_task_basic_info>.size)/4
        
        let kerr: kern_return_t = withUnsafeMutablePointer(to: &info) {
            $0.withMemoryRebound(to: integer_t.self, capacity: 1) {
                task_info(mach_task_self_,
                         task_flavor_t(MACH_TASK_BASIC_INFO),
                         $0,
                         &count)
            }
        }
        
        if kerr == KERN_SUCCESS {
            return Int64(info.resident_size)
        } else {
            return -1
        }
    }
}